USE [Population];

 /* THE PURPOSE OF THIS MODEL:

  This model was created to inform public policy about to genetically safe ways of limiting overpopulation.
  Several methods of limiting the population can be modeled:  

  A: A system of limiting the total number of children per family by law.  
  B: A system of limiting the total number of family lineages through family licensing or family medallions.
  C: Cultural changes that seek to limit the population by changing attitudes about family size while making large families difficult economically. 
  D: A combination of the above approaches.

  The model explores the very long-term genetic effects of each policy on the population and seeks to demonstrate which of these policies 
  is best for maintaining the genetic health of the population. 

  In addition, the "natural state" can also be modeled--where the population is kept down by high child mortality rates.
      
  There remain many uncertainties relating to the accumulation of genetic load in humans and how it is removed through natural selection.
  The model attempts to parameterize these uncertainties, such that different possible scenarios can be modeled, 
  such that a policy can be formed that is safe for many different contingencies about the uncertain nature of human genetics.


 ********************************************/  

 /* WHAT THIS MODEL DOES:
 
  This tracks the genetic load of a population--accross up to 32767 different possible genes, up to 255 variants of each deleterious mutation can be tracked.

  A genome with configurable properties is generated at the start and remains consistent through each run of the simulation.

  A population of 50,000 or so can be simulated over many thousands of years, given a few hours to run on a typical desktop computer.

  Each person passes on genes to their offspring, plus new mutations, according to the principles of Mendellian genetics. 

  By changing the settings of the model, different legal, cultural and genetic scenarios can be simulated to determine the effects of different social policies for different
  contingencies about the nature of human genetics. 
  
  For removing genetic load, four types of natural selection are included into the model, and the genes may be configured have different effects
  on these methods of natural selection. The four methods of natural selection modeled are: 

  A: Child survival selection: Those with high genetic load may not survive childhood.
  B: Adult success selection: Problems such as infertility or economic difficulties could lead to genetic death for adults.
  C: Sexual selection: Individuals will preferentially pick mates that are more "attractive". This could reduce genetic load if attractiveness correlate with fitness.
  D: Inheritance selection: Children compete with each other to receive an inheritance from their parents. Children who succeed in this may have lower genetic load. 

  For child survival and adult success selection, fitness is calculated using a percentage chance of "dying" every year, with this random chance being modified by genes.
  The effects of synergistic epistasis can be modeled and configured to be high or low by modifying the @codeleteriousness setting. 

  For sexual selection and inheritance selection, each person is given scores based on their genes, plus a configurable luck factor,
  and then these values are directly compared to others who they are in competition with, with the best score winning. 
  For these types of selections, since success is determined by comparison between people rather than in a probabilistic way with each mutation increasing chance of death,
  these could potentially be more efficient for removing genetic load--or less efficient if attractiveness and endearment does not correlate well with genetic health. 

  */

    
 /*******************************************************
  * CULTURAL, LEGAL, AND ENVIRONMENTAL SETTINGS SECTION *
  *******************************************************
  These settings model hypothetical differences in laws, culture, medical technology, and the environment. 
  Change these settings to account for different social scenarios. 
  ******************************************************/

 
DECLARE @adult_mort_factor BIGINT = 200;
DECLARE @child_mort_factor BIGINT = 200;
/* factors representing how likely people are to die based on the medical knowledge of the population.
   If 0, no one will die. If 10000, 5% of people with zero genetic load will die in infancy (at or before age 2). 
   Scales linearly. 
   Note: an actuarial table, called the "death_table" is also important for determining who will die.
   Inserts for this table can be found and modified in the "Reset Data.sql" script. 
   For adults, "Death" should not be taken literally. 
   Anything that prevents further reproduction, such as infertility or economic failure would be considered a genetic death. */

DECLARE @adult_mort_increase INT = 0;
DECLARE @child_mort_increase INT = 0;
 /*how much the mort factors above increase everytime a population threshold is crossed.
   this can be used to limit the population in the "natural" state without using hard caps.
   If these is set, survival and success becomes more difficult as the population gets higher.  */
 
DECLARE @death_factor_threshold INT = 2200;
 /*The mort factors can change everytime the population crosses a threshold that is a multiple of this setting.
   The mort factors increase as the population goes higher, or decreases as it goes lower.
   Cannot be zero. */

 DECLARE @legal_child_count INT = 2;
 /*legal number of children per family if there exists a government policy restricting family size. 
   Families will stop having children at this number if the population is above the @legal_poulation threshold.
   Should be set to some number (such as 2).
   if no cap, set to NULL. */

DECLARE @legal_population INT = 13000;
 /*number of people that the government prefers to have (total population) if there exists a policy limiting family size for this population.
   family size restrictions only apply when the population is above this number.
   Note: people over 50 aren't modeled for performance reasons, and so aren't included in this number. 
   if family size is not restricted, set this to NULL*/
  
 DECLARE @track_family_lineages BIT = 1;
 /*if this is 1, families will make an effort to track and maintain unique family lineages for the purposes of inheritance. 
   if this is 0, all matchmaking goes into the same pool and all inheritance is divided evenly between children. 
   If this is 0, @inheritance_selection does nothing and @sexual_selection is less effective. */

DECLARE @legal_family_count INT = null;
 /*should be set if there exists a government policy limiting the total number of family lineages through family licensing or medallions.
   could also be set if there exists some kind of economic restrictions that in practice limit the number of families, 
   such as a limited number of farm plots that could be inherited. 
   if no cap, set to NULL.  
   @trackFamilyLineages must be turned turned to 1 if this is set to a number. */

DECLARE @unlicensed_families BIT = null;
 /*should be set to 1 if it is possible to have a family outside of the family licensing system. 
   these families would have a different birth rate.
   Could be NULL if there is no family licensing system. */


 DECLARE @inheritance_selection BIT = 1;
 /*if this is 1, the more "endearing" children will be given priority for inheritance.
   if this is 0, the oldest surviving child always continues the family lineage.
   inheritance selection is one way that populations can reduce genetic load over time.
   endearment is based on genetics and correlated other kinds of fitness. 
   requires @trackFamilyLineages = 1. */

 DECLARE @sexual_selection BIT = 1;
 /*if this is 1, the more "attractive" people will be more likely to find a marriage.
   if this is 0, people match at random.
   sexual selection is one way that the population can reduce genetic load over time.
   requires @track_family_lineages set to 1 and/or @percentage_matching to be a low number for this to have any practical effect. */

 DECLARE @percentage_matching INT = 50;
 /*The percentage of people in the non-inheritor matchmaking pool that will find a match in any given 2-year period.
   setting this to a lower value will increase the effectiveness of @sexual_selection for couples outside the inheritance matchmaking pool */

 DECLARE @genetic_testing_breeding BIT = 0;
 /*if this is 1, then inheritance selection and sexual selection will use the count of total mutations, including for nonexpressed recessive genes. 
   this type of selection is much more effective for removing low-deleterious mutations or large numbers of mutations.
   The genetic knowledge to implement this kind of breeding program does not currently exist, but may exist in the future. */

 DECLARE @min_cultural_child_count INT = 15;
 DECLARE @max_cultural_child_count INT = 15;
 /*the range in the number of children a typical family prefers to have according to the culture. 
   Upon marriage, each couple will set out to have some number of children within this range.
   They will have a child every two years until they "die" or until this number is reached.
   Note: "Death" in this model should not be taken literally--anything that would prevent further reproduction, 
   such as infertility or economic difficulties, could be considered a genetic death. */

 DECLARE @min_unlicensed_child_count INT = 15;
 DECLARE @max_unlicensed_child_count INT = 15;
 /*the range in the number of children per generation of a unlicensed family lineage.
   does nothing unless both @legal_family_count and @unlicensed_families is set */

 DECLARE @min_age_of_marriage INT = 18;
 /*Minimum age of marriage. Also the typical age of marriage.
   it is recommended to not set this lower than 18 or you will have to rewrite the actuarial table. 
   (by default, the actuarial table has a spike in "death" at age 18 to represent infertility or economic problems in adulthood.) */

 DECLARE @free_marriage_age INT = 20;
 /*Minimum age of marriage for those outside inheritor pool if @trackFamilyLineages is turned on. 
   it is recommended that this value be equal or higher than @min_age_of_marriage or else inheritors won't find a good match, 
   since their best possible suitors would have married each other by then. */

 DECLARE @too_old_for_marriage_age INT = 26
 /*People this age and over will have a harder time finding matches unless they are an inheritor. */

 DECLARE @min_competition_age INT = 14;
 /*must be at least this old to compete with adult children for inheritance.
   used by @inheritance_selection. */
 
 DECLARE @matchmaking_luck_variance INT = 20;
 /*how much luck, rather than genetics, plays into whether or not a person finds a mate.   
   a random value between 0 and this percentage times @surv_comparison_small is added to the total inher_pen of each person. */

 DECLARE @inheritance_luck_variance INT = 20;
 /*how much luck, rather than genetics, plays into whether or not a person inherits their family's household.
   a random value between 0 and this percentage times @surv_comparison_small is added to the total inher_pen of each person. */

 DECLARE @survival_correlation DECIMAL(10,2) = 90;
 /*how much a gene's effect on survival-fitness correlates with the other metrics of fitness. 
   if 100 for each of the four correlation values, these are the same thing.
   if 0, they are completely independent, and the variance in surv_pen has no connection to the other metrics. */

 DECLARE @adult_success_correlation DECIMAL(10,2) = 90;
 /*how much a gene's effect on likelihood of success in adulthood correlates with other metrics of genetic fitness. 
   if 100 for each of the four correlation values, these are the same thing.
   if 0, they are completely independent, and the variance in success_pen has no connection to the other metrics. */

 DECLARE @attractiveness_correlation DECIMAL(10,2) = 90;
 /*how much a gene's effect on attractiveness for the purpose of sexual selection correlates with other metrics of genetic fitness. 
   if 100 for each of the four correlation values, these are the same thing.
   if 0, they are completely independent, and the variance in attr_pen has no connection to the other metrics. */

 DECLARE @endearment_correlation DECIMAL(10,2) = 90;
 /*how much a gene's effect on inheritance-likelihood correlates with other metrics of genetic fitness. 
   if 100 for each of the four correlation values, these are the same thing.
   if 0, the variance in inher_pen has no connection to the other metrics. */


 /**************************************
  *        GENETICS SETTINGS           *
  **************************************
  
  These parameters model uncertainties in how human genetics works.
  Further research and improved knowledge of genetics may allow us to fin dmore accurate values for these parameters.
  The population can be modeled under different contingencies for what these parameters turn out to be. 
  NOTE: If these settings are changed, the simulation should be restarted from year zero by running the "Reset Data.sql" script.
  
  **************************************
 */
 
 DECLARE @mutation_rate DECIMAL(10,2) = 2.1;
 /*Average number of new deleterious mutations per newborn per generation.
   this is by far the most important factor in determining how fast genetic load accumulates.
   the estimate of 2.1 per newborn per generation is taken from this scientific paper: http://www.genetics.org/content/191/4/1321
   other scientific papers give other estimates, so the value is configurable. 
   counterintuitively, the total number of mutations is far more important than how deleterious each is.
   this is because weaker mutations tend to linger in the gene pool for more generations, while stronger mutations are removed more quickly.
   so even though weaker mutations have less effect, their increased duration makes them comparable to even the most deleterious of mutations. 
   NOTE: the mutation rate for newborns is believed to increase based on their parent's age. 
   In "ResetData" there are inserts for a table "parentalagemutations" where the mutationrate is increased based on the age and sex of each parent.
   by default (unless that table is modified), the mutation rate is constant until age 30 for both sexes, and then it increases by .1 per year. */

 DECLARE @codeleteriousness DECIMAL(10,2) = 2.5;
 /*An exponential value which causes deleterious mutations to be more deadly in combination than alone. 
   This is a measure of "synergistic epistasis" in the genome. 
   Increasing this exponent will increase the efficiency of natural selection, and also speed at which genetic load becomes noticable in a population.
   An important open question in genetics is how different mutations interact with each other 
   to potentially cause exponentially more deleteriousness in combination than alone.
   This exponent can be used to model different scenarios of mutation interaction--the higher the value, the more interaction between deleterious mutations.
   If codeleteriousness is high, it will take less generations before accumulation of genetic load begins to have disasterous effects on a population.
   If codeleteriousness is low, then natural selection is less efficient, 
   and high mutation rates may not be survivable in the long term without exceptionally high family sizes.  */
 
 DECLARE @genes_needed_for_lethal BIGINT = 5
 /*a weighting for how many bad genes it takes to kill a person in the "natural" (pre-modern medicine) state. 
   This is the approximate number of median-deleteriousness fully-expressed (homozygous) mutations that a person would need 
   to have to die 100% of the time if @child_mort_factor is 10000 */
  
 DECLARE @genome_size SMALLINT = 20000;
 /*total number of genes in the genome that are considered relevant. 
   the model can handle up to 32767 genes.
   this represents the number of things that can break as far as deleterious mutations are concerned. 
   practical effects of changing this setting: The more possible genes there are, the less likely any recessive mutations are to line up with each other. 
   this means it will take longer and more recessive mutations will build up before causing problems.
   This pushes the total number of recessive mutations in the genome to a higher equilibrium. */
 
 DECLARE @lethality_variance INT = 30;
 /*variance in how deleterious each mutation is. 
   The more variance in this value, the harder it is to purge deleterious mutations through direct comparison methods. 
   One common error made in genetics models is not accounting for this property of mutations.
   If all mutations were equally deleterious, then sexual selection and similar methods of relative selection would be
   far more efficient in removing genetic load.*/

 DECLARE @lethality_minimum INT = 0
 /*minimum deleteriousness of each mutation. This number can be negative to sometimes generate beneficial mutations.
   If too many mutations are beneficial, they will build up in the genome as they become fixed, causing the model to run slower.
   This will also confuse the other settings of the model by altering the survival rates in unpredictable ways,
   so it is recommended not to make mutations too beneficial, and instead
   allow the default (non-mutated) version of the gene to be the best variant possible for the particular environment.
   Whether a mutation is beneficial or not also depends on the environment and the type of natural selection that the person is under.
   There are an infinite number of possible environments and types of selection pressure, 
   and the genes that are successful in each type of environment are highly correlated with each other. 
   For simplicity, only four of these infinite types of selection are modeled. */

 DECLARE @variant_variance INT = 10;
 /*variance in how deadly each variant of each mutation is. A number between 0 and this number is added to @lethality_variance to determine the
   penalty of the most deadly variant of the gene. */
 
 DECLARE @lethality_exponent DECIMAL(10,2) = 1.7;
 /*This adds an additional, exponential component to variance. 
   rather than the deleteriousness of all genes being on a linear scale, this causes the most deleterious mutations to pack an additional punch. */
 

 DECLARE @proportion_purely_recessive DECIMAL(10,2) = 20;
 /*Percentage (out of 100) of genes that are purely recessive and show no effect in heterozygous form.
   the more genes are recessive, the longer it takes to purge them from the genome and also the longer it takes them to cause any problems.
   after enough time passes, the number of bad recessive genes hits an equilibrium state where they are increasingly likely to be expressed. */

 DECLARE @proportion_beneficial_in_hetero DECIMAL(10,2) = 0;
 /*Percentage (out of 100, not including purely recessive genes) that show a benefit rather than a detriment in heterozygous form
   These mutations are still deleteriuos in homozygous form. 
   some genes, such as the gene that causes sickle cell anemia, can actually show a benefit in heterozygous form. 
   this setting allows the user to add genes like this to the genome.
   This adds a large random factor, so this setting should generally be turned off, but if it is turned on, you can see that these genes
   will reach an equilibrium state in the genome, which alters the baseline survivability of each person in a way which is related to
   the total size of the breeding pool, but otherwise has little effect on the conclusions of the model. */
  
 DECLARE @heterozygous_benefit_variance INT = 0;
 /*Some genes may give benefits in heterozygous form, even as they are deleterious in homozgyous form. This is the maximum benefit for these genes. */

 DECLARE @heterozygous_divisor DECIMAL(10,2) = 20;
 /*homozygous deleterious mutations are often stronger than the same mutation in heterozygous form. 
   the deleteriousness of a heterozygous mutation is divided by a random value between 1 and this number. */

 DECLARE @fix_mutation_chance INT = 600
 /*relates to the chances that a random mutation will completely revert a previous mutation if it strikes the same gene number and variant number, 
   Set to 0 if this is impossible. 
   For example: Let's say there are 20,000 genes in the genome, and
   a new child inherits one deleterious mutation from their parents, plus one newly generated mutation.
   What are the odds that these two mutations cancel out and eliminate each other, resulting in a deleterious-mutation-free genome?
   In this model, the chances are:  1/20,000 for the new mutation hitting the same location as the old mutation, 
   multiplied by  1/256 for it rolling the same variant, 
   then 1 divided by the above number, since 1 in 256*20,000 would be too often. 
   This comes out to 1 in 3,072,000,000 if the above number is 600, which is about equal to the number of base-pairs in the human genome. 
   This implies that with a population of 512,000 which has a fixed deleterious mutation and a mutation rate of 2 per generation, 
   they would be expected to re-evolve the non-deleterius version of that gene once every 3000 generations or so. */

 --TODO: IMPLEMENT THIS FEATURE
 --DECLARE @homozygous_lethal_in_utero DECIMAL(10,2) = 0
 /*percentage of genes that are deleterious in heterozygous form and completely lethal in utero in homozygous form.
   the couple gets X tries to conceive a child without any of these genes homogenous or they won't have a child in this two-year period.
   NOTE: This may be implemented in a later version. 
   it is expected that these mutations would be less important than other mutations, since they would be removed more efficiently.
   if there are believed to be a lot of these types of genes, this feature could be implemented or the deleterious mutation rate could be lowered to compensate. */


 /*************************************** 
  *         SIMULATION SETTINGS         *
  ***************************************
  How long the model runs and how often it reports data and how many people it can handle
  **************************************/
 
 DECLARE @report_frequency INT = 50;
 /*How often (every x years) the model reports data to the SQL Server interface
   NOTE: The model iterates once every two years. So this must be an even number. */

 DECLARE @years_simulated INT = 500;
 /*Number of years the model should simulate.
   NOTE: The model iterates once every two years. So this must be an even number. */

 DECLARE @max_population INT = 200000;
 /*total number of people that can be modeled. Hard Cap for performance reasons.
   it is recommended that the other parameters of the model be set in such a way that this is never reached. 
   note: people over age 50 aren't modeled for performance reasons, so they aren't included in this number. 
   note: if this is set too low and the number of mutations per person is high, SQL Server Express might hit its 10 gigabyte limit and the model will stop abruptly. */

 DECLARE @clean_tables_frequency INT = 200;
 /*how often the mutation tables are cleaned out of old data.
   cleaning them out more often will save hard disk space and make it possible to model larger numbers of people and mutations before hitting the 10 gig limit for SQL Express.
   cleaning them out less often will (to a point) cause the model to run faster.
   NOTE: The model iterates once every two years. So this must be an even number. */

 /* END SETTINGS SECTION */





 /************************************
  *   INTERNAL VARAIBLES SECTION     *
  ************************************
  These variables are used internally by the model and should not be modified by hand 
  ***********************************/
 
 DECLARE @infant_mortality_factor INT = (SELECT death_rate 
                                           FROM death_table 
                                          WHERE age = 2);
 
  /*@surv_comparison_big and @surv_comparison_small: These two values are used to calculate the relative likelyhood that each point of surv_pen will kill the person. 
   the values are weighted against the overall amount of deleteriousness each gene produces, 
   so that modifying the genomic settings such that average surv_pen is lower or higher won't radically alter the natural-state death rate. 
   the values are balanced such that if the @child_mort_factor_current is 10000, a child with zero genetic load will have a 5% chance of dying in infancy, 
   but a child with approximately @genes_needed_for_lethal fully expressed median-deleteriousness bad genes will have a 100% chance of dying in infancy. */
 DECLARE @surv_comparison_big BIGINT = Power(@genes_needed_for_lethal*Power(@lethality_variance/2 + 1 + @variant_variance/4, 
                                             @lethality_exponent),
                                       @codeleteriousness)
                                       *10000*@infant_mortality_factor; 
 
 DECLARE @surv_comparison_small BIGINT = @surv_comparison_big / 20 / 10000 / @infant_mortality_factor

 
 /*retrieves the current year from the statistics table to begin where the simulation previously left off. */
 DECLARE @current_year INT = (SELECT Max(current_year)+2 
                                FROM statistics_table);

 /*the simulation will stop once the current year reaches the stop_year */
 UPDATE stop_year
    SET stop_year = @current_year + @years_simulated;

 /*declaring variables that will be used later. */
 DECLARE @current_pop INT;
 DECLARE @last_legal_lineage INT;
 DECLARE @min_id INT;
 DECLARE @family_count INT = -2147483640;
 DECLARE @groom_overflow INT = -1;
 DECLARE @new_matches INT;
 DECLARE @child_mort_factor_current BIGINT;
 DECLARE @adult_mort_factor_current BIGINT;
 DECLARE @i INT = 1;
 DECLARE @save_identity INT;
 
 /* If this is a new simulation, randomly generates a new genome from the genomic settings. */
 IF @current_year = 0
 BEGIN
 
  /*Randomly generates a genome according to the genomic settings. 
    Each iteration of the loop generates a new gene */
  WHILE @i <= @genome_size
  BEGIN
   
   WITH
   /*this subquery generates some random values for determining the importance of this gene */
   genome_randomizer (base_penalty, dominance_type, base_penalty_roll)
   AS
   (
   SELECT /*randomly generates the basic fitness penalty of the mutated version of this gene */
          @lethality_minimum+Floor(Rand(Checksum(NewID()))*@lethality_variance) AS base_penalty,
          
          /*a random algorithm determines whether the mutated version of the gene is purely recessive or not,
            and whether or not this gene shows unexpected benefits in heterozygous form. */
          CASE WHEN floor(rand(checksum(newid()))*100)+1 <= @proportion_purely_recessive 
          THEN 1 /* 1 for purely recessive mutations */
          WHEN floor(rand(checksum(newid()))*100)+1 <= @proportion_beneficial_in_hetero
          THEN 2 /* 2 for genes that are beneficial in heterozygous form but not in homozygous form */
          ELSE 0 /* 0 for dominant mutations */
          END AS dominance_type,

          /* a random roll that determines which of the four types of natural selection the base_penalty will best represent.
             the algorithm is designed such that the deleteriousness of each mutation is correlated accross selection types. 
             the correlated component is the base_penalty*/
          floor(rand(checksum(newid()))*4) as base_penalty_roll
   ),
   
   /* this subquery generates more random values and applies correlations from the genome_randomizer to determine the fitness penalty of each mutation. */
   genome_subquery (mutation_id, base_surv_pen, base_attr_pen, base_inher_pen, base_success_pen, dominance_type, variance, heterozygous_divisor, heterozygous_benefit)
   AS
   (
   SELECT /*uniquely identifies the gene where the mutation is located */
          @i as mutation_id,
       
          /*how important this gene is to child survival.*/
          CASE WHEN base_penalty_roll <> 0 
          THEN base_Penalty*@survival_correlation/100 + 
               Floor(Rand(Checksum(NewID()))*@lethality_variance) * 
               (100-@survival_correlation)
               /100 
          ELSE base_penalty END AS base_surv_pen,
         
          /*how important this gene is to attractiveness for the purpose of sexual selection */
          CASE WHEN base_penalty_roll <> 1 
          THEN base_penalty*@attractiveness_correlation/100 + 
               Floor(Rand(Checksum(NewID()))*@lethality_variance) * 
               (100-@attractiveness_correlation)
               /100 
          ELSE base_penalty END AS base_attr_pen,

          /*how important this gene is to fertility or to financial success in adulthood */
          CASE WHEN base_penalty_roll <> 2 
          THEN base_penalty*@adult_success_correlation/100 + 
               Floor(Rand(Checksum(NewID()))*@lethality_variance) * 
               (100-@adult_success_correlation)
               /100 
          ELSE base_penalty END AS base_success_pen,

          /*how important this gene is to endearment to family for the purposes of securing an inheritance */
          CASE WHEN base_penalty_roll <> 3 
          THEN base_penalty*@endearment_correlation/100 + 
               floor(rand(checksum(newid()))*@lethality_variance) * 
               (100-@endearment_correlation)
               /100 
          ELSE base_penalty END AS base_inher_pen,

          dominance_type as dominance_type,

          /* a random value that determines how much each variant of the mutation differs in power. 
             variants of each mutation are numbered from 1 to 255, with 1 representing the least deleterious possible version,
             and 255 representing the most deleterious variant. */
          Power(Rand(Checksum(NewID()))*(@variant_variance+1), @Lethality_exponent) as variance,

          /*homozygous deleterious mutations are often stronger than the same mutation in heterozygous form. 
            the deleteriousness of a heterozygous mutation is divided by a random value between 1 and the @heterozygous_divisor setting. */
          CASE WHEN Dominance_type = 0 THEN 1+rand(checksum(newid()))*(@heterozygous_Divisor) ELSE null END as Heterozygous_Divisor,

          /*if the gene gives a benefit in heterozygous form, the magnitude of the benefit is randomly generated here */
          CASE WHEN Dominance_Type <> 2
          THEN null
          ELSE power(rand(checksum(newid()))*(@Heterozygous_Benefit_Variance+1),@Lethality_exponent) END as Heterozygous_Benefit
          FROM genome_randomizer
   )

   /*inserts the data about mutations for this gene into the genome table*/
   INSERT INTO genome (mutation_id, base_surv_pen, base_attr_pen, base_inher_pen, base_success_pen, dominance_type, variance, heterozygous_divisor, heterozygous_benefit)
   SELECT mutation_id, 

          /*the deleteriousness of each mutation is taken to the power of the @lethality_exponent setting.
            this setting causes the variance in deleteriousness to be on an exponential scale rather than a linear one */
          CASE WHEN base_surv_pen >= 0 
          THEN Power(base_surv_pen,@lethality_exponent)
          ELSE -Power(-base_surv_pen,@lethality_exponent) END AS base_surv_pen,

          CASE WHEN base_attr_pen >= 0 
          THEN Power(base_attr_pen,@lethality_exponent)
          ELSE -Power(-base_attr_pen,@lethality_exponent) END AS base_attr_pen,

          CASE WHEN base_inher_pen >= 0 
          THEN Power(base_inher_pen,@lethality_exponent)
          ELSE -Power(-base_inher_pen,@lethality_exponent) END AS base_inher_pen,

          CASE WHEN base_success_pen >= 0 
          THEN Power(base_success_pen,@lethality_exponent)
          ELSE -Power(-base_success_pen,@lethality_exponent) END AS base_success_pen,

          dominance_type,

          variance,

          heterozygous_divisor,

          heterozygous_benefit
     FROM genome_subquery;
   
   /*increment @i to prepare to move onto the next possible mutation for this genome */
   SET @i = @i + 1;

  END;
 END;
 

 /********************************************
  *           BEGIN SIMULATION               *
  ********************************************
  The simulation works as a loop where deaths, then births, then marriages occur. Each loop represents a timespan of two years.
  The simulation can be stopped in mid-progress by running this script: 
  UPDATE stop_year SET stop_year = 0 
  *******************************************/

 /*the simulation continues until a set year */
 WHILE @current_year < (SELECT stop_year FROM stop_year)
 BEGIN 

  /* **************************************************************************** */
  /* deaths section -- removes any people that did not survive the last two years */
  /* **************************************************************************** */

  /*updates the death rate to account for the population being over certain threshholds.
    as these thresholds are crossed, the death rate may increase. */
  SET @child_mort_factor_current = @child_mort_factor + 
                                   @child_mort_increase *
                                   ((SELECT current_pop+births 
                                       FROM statistics_table 
                                      WHERE current_year = @current_year - 2) /
                                   @death_factor_threshold);
  
  /*adult "mortality" should not be taken literally. Anything that would prevent further reproduction would be considered a genetic death. */
  SET @adult_mort_factor_current = @adult_mort_factor + @adult_mort_increase *
                                   ((SELECT current_pop+births 
                                       FROM statistics_table 
                                      WHERE current_year = @current_year - 2) /
                                   @death_factor_threshold);

  /*each person has a random chance of dying every year, based on the actuarial table, penalized by their "surv_pen" stat.
    people that died are first logged in the staging_dead_person table.
    people 50 or older are always removed from the model for performance reasons. */
  INSERT INTO staging_dead_person (person_id)
       SELECT p.person_id
         FROM person p
         JOIN death_table dt 
           ON p.year_of_birth = @current_year - dt.age
        WHERE floor(rand(checksum(newid()))*@surv_comparison_big) < 
              CASE WHEN dt.age < 18 
              THEN @child_mort_factor_current*death_rate*(@surv_comparison_small+surv_pen) 
              ELSE @adult_mort_factor_current*death_rate*(@surv_comparison_small+success_pen) END 
              or dt.age = 50;

  /*marriage records that need to be dissolved are inserted into the staging_done_marriedpair table */
  /* for dead wives: */
  INSERT INTO staging_done_married_pair (pair_id, widow_id)
       SELECT pair_id, mp.male_id
         FROM married_pair mp
         JOIN staging_dead_person sdp
           ON sdp.person_id = mp.female_id;

  /*for dead husbands: */
  INSERT INTO staging_done_married_pair (pair_id, widow_id)
       SELECT pair_id, mp.female_id
         FROM married_pair mp
         JOIN staging_dead_person sdp
           ON sdp.person_id = mp.male_id;
  
  /*remarriage is not programmed into this model, so all widow(er)s can drop dead. RIP. */
  INSERT INTO staging_dead_person (person_id)
       SELECT person_id 
         FROM person p 
         WITH (forceseek ([clustered_index_person](person_id)))
         JOIN staging_done_married_pair sdmp  
           ON sdmp.widow_id = p.person_id;

  /*average number of offspring of people who died this year. Should be 2 in a stable population.
    used to calculate the fertility_variance. */ 
  DECLARE @average_fertility INT = (
   SELECT Avg(Cast(IsNull(mpf.children,0) + IsNull(mpm.children,0) AS DECIMAL(10,4)))
     FROM staging_dead_person sdp
LEFT JOIN married_pair mpf 
       ON mpf.female_id = sdp.person_id
LEFT JOIN married_pair mpm 
       ON mpm.male_id = sdp.person_id);

  /*statistics recorded. */
  INSERT INTO statistics_table (deaths, current_year, current_pop, fertility_variance)

       SELECT /* number of people who died in the last two year period. */
              count(DISTINCT sdp.person_ID) AS deaths, 

              /* the current year */
              @current_year AS current_year, 

              /* calculates the current pop for the year based on last's pop plus last's births minus the people who just died */
              (SELECT IsNull(last_year.current_pop + last_year.births,0) 
                 FROM statistics_table last_year
                WHERE last_year.current_year = @current_year -2) - 
              Count(DISTINCT sdp.person_id) AS current_pop,

              /* of all the people who died this year, calculates the average of how much their fertility differed from the average fertility */
              Avg(Cast(CASE WHEN IsNull(mpf.children,0) + IsNull(mpm.children,0) >= @average_fertility 
                       THEN IsNull(mpf.children,0) + IsNull(mpm.children,0) - @average_fertility
                       ELSE @average_fertility - IsNull(mpf.children,0) - IsNull(mpm.children,0) END AS DECIMAL(10,4))) AS fertility_variance

         FROM staging_dead_person sdp
    LEFT JOIN married_pair mpf 
           ON mpf.female_id = sdp.person_id
    LEFT JOIN married_pair mpm 
           ON mpm.male_id = sdp.person_id
    
  /* marriages where one of the members died are dissolved. */
  DELETE mp 
    FROM married_pair mp
   WHERE mp.pair_id in (SELECT pair_id 
                          FROM staging_done_married_pair);
     
  /* dead people are removed from the person table */
  DELETE p 
    FROM person p
   WHERE p.person_id in (SELECT person_id 
                           FROM staging_dead_person);
 
   /*Updates the child mortality statistics. 
     Calculated by comparing the number of sixteen-year-olds alive
     with the number of people who were born sixteen years ago. */
  UPDATE statistics_table 
     SET child_mortality = (-(SELECT 100*Cast(Count(*) AS DECIMAL(10,4)) 
                                FROM person 
                               WHERE year_of_birth = @current_year-16) /
                            NullIf((SELECT births 
                                      FROM statistics_table 
                                     WHERE current_year = @current_year-16),
                                   0) + 
                            100)
   WHERE current_year = @current_year;
  
  

  /*the staging tables are truncated to be re-used fresh next iteration. */
  TRUNCATE TABLE staging_dead_person;
  TRUNCATE TABLE staging_done_married_pair;

  /*the mutation tables are cleaned out once every so often, so they don't get too big. */
  IF @current_year%@clean_Tables_Frequency = @clean_Tables_Frequency - 2
  BEGIN

   /*Since the number of deletes would be well over half the data, it's faster to just recreate the tables from scratch. 
     First, we insert only the data that is still relevant into staging tables. */
   /*Current data from the mutation table is inserted into staging_mutation: */
   INSERT INTO staging_mutation (person_id, mutation_id, mothers_variant, fathers_variant)
        SELECT person_id, mutation_Id, mothers_variant, fathers_variant
          FROM mutation
         WHERE person_id in (SELECT person_id 
                               FROM person);

   /*Current data from the married_pair_mutation table is inserted into the staging_married_pair_mutation table */
   INSERT INTO staging_married_pair_mutation (pair_id, mutation_id, females_mothers_variant, females_fathers_variant, males_mothers_variant, males_fathers_variant)
        SELECT                                pair_id, mutation_id, females_mothers_variant, females_fathers_variant, males_mothers_variant, males_fathers_variant
          FROM married_pair_mutation
         WHERE pair_id in (SELECT pair_id 
                             FROM married_pair);

   /*swaps the names of the staging tables and the old tables */
   EXEC sp_rename 'mutation', 'old_mutations';  
   EXEC sp_rename 'staging_mutation', 'mutation';
   EXEC sp_rename 'old_mutations', 'staging_mutation'
   EXEC sp_rename 'married_pair_mutation', 'old_pair_mutations';  
   EXEC sp_rename 'staging_married_pair_mutation', 'married_pair_mutation';
   EXEC sp_rename 'old_pair_mutations', 'staging_married_pair_mutation'

   /*the old tables are truncated. */
   TRUNCATE TABLE staging_mutation;
   TRUNCATE TABLE staging_married_pair_mutation;

   /*since we're doing a lot of inserts, we'll want to update the statistics for tables every so often so SQL Server knows how to make good query plans. */
   EXEC sp_updatestats;

  END;
  

  /* **************************************************************** */
  /* births section -- children are born to couples                   */
  /* **************************************************************** */
      
  /*get the current total number of people from the statistics table for use in determining whether we have reached a population limit. */
  SELECT @current_pop = current_pop 
    FROM statistics_table 
   WHERE current_year = @current_year;
  
  /*if a family size laws limits the number of children each pair can plan to have without a special license, 
    then update the number of children each family plans to have based on the law.   */
  IF @current_pop > @legal_population and @legal_child_count is not NULL
   UPDATE married_pair 
      SET desired_children = @legal_child_count
    WHERE desired_children > @legal_child_count;

  /*each couple produces one child every two years if able until they have as many as they want or can legally have.
    new children are first inserted into the staging_person table for processing. */
  IF @current_pop < @max_population
   INSERT INTO staging_person (parents_id, lineage_id, new_mut_count, line_gen, inheritor)
        SELECT TOP (@max_population - @current_pop)

               /*uniquely identifies the child's parents, based on their marriage record */
               mp.pair_id AS parents_id,

               /*the family lineage the child belongs to */
               mp.lineage_id AS lineage_id,

               /*a non-deterministic function to calculate how many new mutations the child should receive, based on the mutation rate. */
               dbo.Calculate_Mut_Count(100*(@mutation_rate +
                                            isnull(pa_mother.increased_mutation_rate,0) +
                                            isnull(pa_father.increased_mutation_rate,0))) as newmutcount,
               
               /* uniquely identifies the generation of the family lineage that the child belongs to. */
               mp.line_gen +1 as line_gen,

               /* indicates whether or not this family keeps a record of lineages and inheritances,
                  or participates in any family licensing system. */
               CASE WHEN mp.track_lineages = 1 and @track_Family_Lineages = 1 
               THEN 0 ELSE null END AS inheritor
                
          FROM married_pair mp
          JOIN person mother 
            ON mother.person_id = mp.female_id
          JOIN person father 
            ON father.person_id = mp.male_id
     LEFT JOIN parental_Age_Mutations pa_mother 
            ON mother.year_of_birth = @current_Year - pa_mother.age and pa_mother.sex = 0
     LEFT JOIN parental_Age_Mutations pa_Father 
            ON father.year_of_birth = @current_Year - pa_father.age and pa_father.sex = 1
         WHERE mp.children < mp.desired_children
      ORDER BY floor(rand(checksum(newid()))*10000); 
  
  /*New mutations are inserted into the mutation table for the person.
    First, mutations are randomly inherited from parents according to mendellian principles. */
  WITH staging_mutation (person_id, mutation_id, mothers_variant, fathers_variant)
  AS
  (
  SELECT c.person_id as person_id,
         mpm.mutation_id as mutation_id,

         /* for each gene, the child will inherit a random copy from their mother. This is their mothers_variant */
         CASE WHEN floor(rand(checksum(newid()))*2) = 1 
         THEN females_mothers_variant 
         ELSE females_fathers_variant END AS mothers_variant,

         /* for each gene, the child will inherit a random copy from their father. This is their fathers_variant */
         CASE WHEN floor(rand(checksum(newid()))*2) = 1 
         THEN males_mothers_variant 
         ELSE males_fathers_variant END AS fathers_variant

    FROM married_pair_mutation mpm
    JOIN staging_person c 
      ON mpm.pair_id = c.parents_id
  )
  /*the inherited mutations of the child are added to the mutation table */
  INSERT INTO mutation (person_id, mutation_id, mothers_variant, fathers_variant)
       SELECT person_id, mutation_id, mothers_variant, fathers_variant
         FROM staging_mutation
        WHERE mothers_variant <> 0 or fathers_variant <> 0

  /*next, new random mutations are generated. 
    if a mutation at the same loci already exists, the previous mutation is overriden.
    occasionally this may cause the inherited mutations to revert back to the non-mutated state. */
  MERGE mutation AS target  
  USING (SELECT person_id, 
                mutation_id as mutation_id,
              
                /*in the unlikely event that the same mutation is generated twice for the same child, 
                  one of the mutations is overriden by the Max statements below.
                  this is necessary to prevent the optimized merge algorithm from throwing an error in certain edge cases.
                  dealing with these kinds of rare double-mutations in a more natural way is not worth the performance costs,
                  and this has no implications for the conclusions of the model. */
                
                /*randomly decides whether the mutation overrides the mothers_variant or the fathers_variant 
                  rolls 0: it replaces the mothers_variant
                  rolls 1: it replaces the fathers_variant */  
                Max(Floor(Rand(Checksum(NewID()))*2)) as side,

                /*randomly generates the new variant, which represents a deleteriousness scale from 0 to 255 */
                Max(Floor(Rand(Checksum(NewID()))*256)) as new_variant

          FROM (SELECT c.person_id,
                     
                       /* the locus of the mutation is generated randomly */
                       1+Floor(Rand(Checksum(NewID()))*@genome_size) AS mutation_id,

                       /* the number of new mutations that should be generated for the child, calculated earlier. */
                       c.new_mut_count

                  FROM staging_person c ) t
          JOIN small_integers i 
            ON i.n <= t.new_mut_count
      GROUP BY person_id, mutation_id) 
     AS source (person_id, mutation_id, side, new_variant)

     ON (target.mutation_id = source.mutation_id and target.person_id = source.person_id)  

     /* for the highly unlikely event where the mutation reverts the gene to the non-deleterious version,
        causing the mutation to be removed from the dataset. */
     WHEN matched 
          and source.new_variant = 0 
          and (target.mothers_variant = 0 
               or source.side = 0) 
          and (target.fathers_variant = 0 
               or source.side = 1) 
          and @fix_mutation_chance <> 0 
          and floor(rand(checksum(newid()))*@fix_mutation_chance) = 0
     THEN DELETE
 
     /* when the new mutation replaces an existing mutated gene at that locus rather than a healthy gene. */
     WHEN matched 
     THEN UPDATE 

             SET mothers_variant = CASE WHEN source.side = 0 
                                             and (source.new_variant <> 0  
                                                  or floor(rand(checksum(newid()))*@fix_mutation_chance) = 0)
                                   THEN source.new_variant 
                                   ELSE target.mothers_variant END,

                 fathers_variant = CASE WHEN source.side = 1 
                                             and (source.new_variant <> 0                                                  
                                                  or floor(rand(checksum(newid()))*@fix_mutation_chance) = 0)
                                   THEN source.new_variant 
                                   ELSE target.fathers_variant END

      /* for brand new mutations, replacing healthy genes at that location. 
         healthy genes are not stored in the mutation table for performance reasons.
         genetic data is only stored for genes that have mutations on them. */
      WHEN NOT MATCHED 
      THEN INSERT (person_id, mutation_id, mothers_variant, fathers_variant)  
           VALUES (source.person_id, 
                   source.mutation_id, 
                   CASE WHEN source.side = 0 THEN source.new_variant ELSE 0 END, 
                   CASE WHEN source.side = 1 THEN source.new_variant ELSE 0 END);


  WITH
  mutation_type_filter (mutation_type, divisor, variance, base_surv_pen, base_attr_pen, 
                        base_inher_pen, base_success_pen, mothers_variant, fathers_variant, 
                        person_id, heterozygous_benefit)
  AS
  (
  SELECT /* calculates the type of mutation this is, for use later in determining its penalties */
         CASE WHEN (sm.fathers_variant <> 0 and sm.mothers_variant <> 0) 
                                         or (g.dominance_type = 0 
                                             and (sm.fathers_variant <> 0 
                                                  or sm.mothers_variant <> 0))
         THEN 'deleterious'
         WHEN (g.Dominance_Type = 2 and (sm.fathers_variant <> 0 or sm.mothers_variant <> 0))
         THEN 'beneficial'
         ELSE 'recessive' END AS mutation_type,

         /* how much the deleteriousness of this mutation should be reduced from the base amount */
         CASE WHEN sm.fathers_variant = 0 or sm.mothers_variant = 0 
         THEN heterozygous_Divisor ELSE 1 END AS divisor,

         /* calculates the additional penalty from the particular variant of the mutation */
         (g.variance*CASE WHEN (sm.mothers_variant > sm.fathers_variant and g.Dominance_Type <> 0) or 
                               (sm.mothers_variant < sm.fathers_variant and g.Dominance_Type = 0) 
                     THEN fathers_variant
                     ELSE mothers_variant END)/256 AS variance,
         
         base_surv_pen,
         base_attr_pen,
         base_inher_pen,
         base_success_pen,
         mothers_variant,
         fathers_variant,
         c.person_id,
         heterozygous_benefit

    FROM staging_person c
    LEFT JOIN mutation sm on sm.person_id = c.person_id
    LEFT JOIN genome g on g.mutation_id = sm.mutation_id

  ),
  
  /* all the deleteriousness from all mutations for the child are summed together for each of the four selection types */
  genetic_penalty_subquery (surv_pen, attr_pen, inher_pen, success_pen, mutation_count, person_id)
  AS
  (    
  SELECT /* the childhood survival penalties are summed */
         Sum(Cast(CASE WHEN mutation_type = 'deleterious'
                       THEN base_surv_pen / divisor + variance
                       WHEN mutation_type = 'beneficial'
                       THEN -heterozygous_benefit+variance
                       ELSE 0 END AS BIGINT)) as surv_pen,

         /* the attractiveness penalties are summed */
         Sum(Cast(CASE WHEN mutation_type = 'deleterious'
                       THEN base_attr_pen / divisor + variance
                       WHEN mutation_type = 'beneficial'
                       THEN -heterozygous_benefit+variance
                       ELSE 0 END AS BIGINT)) as attr_pen,

         /* the endearment penalties are summed */
         Sum(Cast(CASE WHEN mutation_type = 'deleterious'
                       THEN base_inher_pen / divisor + variance
                       WHEN mutation_type = 'beneficial'
                       THEN -heterozygous_benefit+variance
                       ELSE 0 END AS BIGINT)) as inher_pen,

         /* the adult success penalties are summed */
         Sum(Cast(CASE WHEN mutation_type = 'deleterious'
                       THEN base_success_pen / divisor + variance
                       WHEN mutation_type = 'beneficial'
                       THEN -heterozygous_benefit+variance
                       ELSE 0 END AS BIGINT)) as success_pen,

         /* counts the total number of mutations for use in statistics or if @genetic_testing_breeding is enabled */
         Sum(CASE WHEN mothers_variant <> 0 and fathers_variant <> 0 
                  THEN 2 
                  WHEN mothers_variant <> 0 or fathers_variant <> 0
                  THEN 1
                  ELSE 0 END) as mutation_count,

         person_id

    FROM mutation_type_filter
   GROUP BY person_id
  )
  /*each new child receives a penalty to its survivability and attractiveness based on its genetics. */

  UPDATE c2
  
         /*The @codeleteriousness setting adds the effects of synergistic epistasis. The total penalties are taken to this exponent to 
           model the effects of mutations being more deleteriousness in combination than alone. */
         /*If the child has enough beneficial mutations, its penalty_values may be negative, indicating increased reproductive success over the default genome.
           To avoid ever reducing the possibility of death to zero, there is a soft cap for how much total benefit a child can receive from mutations. 
           In the below implementation, it impossible for the base death rate to be reduced by more than 20% from mutations.*/
     SET c2.surv_pen =  IsNull(CASE WHEN t.surv_pen < 0 
                              THEN -(@surv_comparison_small/5*Power(-t.surv_pen,@codeleteriousness)/(Power(-t.surv_pen,@codeleteriousness)+@surv_comparison_small/5))
                              ELSE Power(t.surv_pen,@codeleteriousness) END,0),

         c2.attr_pen =  IsNull(CASE WHEN t.attr_pen < 0
                           THEN -(@surv_comparison_small/5*Power(-t.attr_pen,@codeleteriousness)/(Power(-t.attr_pen,@codeleteriousness)+@surv_comparison_small/5))
                           ELSE Power(t.attr_pen,@codeleteriousness) END,0),
         
         c2.inher_pen = IsNull(CASE WHEN t.inher_pen < 0
                           THEN -(@surv_comparison_small/5*Power(-t.inher_pen,@codeleteriousness)/(Power(-t.inher_pen,@codeleteriousness)+@surv_comparison_small/5))
                           ELSE Power(t.inher_pen,@codeleteriousness) END,0),

       c2.success_pen = IsNull(CASE WHEN t.success_pen < 0
                           THEN -(@surv_comparison_small/5*Power(-t.success_pen,@codeleteriousness)/(Power(-t.success_pen,@codeleteriousness)+@surv_comparison_small/5))
                           ELSE Power(t.success_pen,@codeleteriousness) END,0),

    c2.mutation_count = IsNull(t.mutation_count,0)

    FROM staging_person c2
    LEFT JOIN genetic_penalty_subquery t on t.person_id = c2.person_id

  /* The child is created and randomly assigned a sex. */
  INSERT INTO person (person_id, parents_id, year_of_birth, lineage_id, inheritor, mutation_count, sex,                             
                      line_gen, surv_pen, success_pen,
                      attr_pen, 
                      inher_pen )
       SELECT         person_id, parents_id, @current_year, lineage_id, inheritor, mutation_count, Floor(Rand(Checksum(NewID()))*2),
                      line_gen, surv_pen, success_pen,
                      CASE WHEN @Genetic_Testing_Breeding = 1 
                      THEN mutation_count 
                      ELSE Floor(Rand(Checksum(NewID()))*@surv_comparison_small*@matchmaking_luck_variance/100)+attr_pen END AS attr_pen,
                      CASE WHEN @Genetic_Testing_Breeding = 1 
                      THEN mutation_count 
                      ELSE Floor(Rand(Checksum(NewID()))*@surv_comparison_small*@inheritance_Luck_Variance/100)+inher_pen END as inher_pen
         FROM staging_person 
  
  /* count new births and add to statistics_table */
  UPDATE statistics_table
     SET births = (SELECT count(*) 
                     FROM staging_person)
   WHERE current_year = @current_year
      
  /*the count of the children the couple has produced is increased by one. */
  UPDATE mp 
     SET mp.children = mp.children + 1
    FROM married_pair mp
    JOIN staging_person sp 
      ON sp.parents_id = mp.pair_id
      
  /* Remembers the next identity seed from staging_person before truncating the table. */
  SET @save_identity = ident_current('staging_person')+1;

  /* If the value is close to causing an integer overflow error, reset it to 1. */
  IF @save_identity > 2000000000
     SET @save_identity = 1;

  /* Clears the staging_person table of data in preparation for the next run */
  TRUNCATE TABLE staging_person;

  /* Resets the identity seed of staging_person to the proper value. */
  DBCC checkident('staging_person', RESEED, @save_identity);
  
  /* **************************************************************** */
  /* matchmaking section -- new couples get together                  */
  /* **************************************************************** */
  
  /*sets the preferred inheritor of each family line, based on either their endearment or on birth order.*/
  IF @track_Family_Lineages = 1
  BEGIN  
   /* if @inheritanceSelection is turned on, the family will pick their inheritor based on endearment. */
   IF @inheritance_Selection = 1
    UPDATE person 
       SET inheritor = 1
     WHERE inheritor = 0 
           and year_of_birth <= @current_year - @min_age_of_marriage
           and not exists (SELECT 1 
                             FROM person p2 
                            WHERE (p2.inher_pen < person.inher_pen
                                   or (p2.attr_pen = person.attr_pen and p2.person_id < person.person_id)
                                   or p2.inheritor = 1)
                                  and p2.lineage_id = person.lineage_id
                                  and p2.line_gen >= person.line_gen
                                  and p2.year_of_birth <= @current_year - @min_competition_age);
   ELSE
   /* if @inheritanceSelection is disabled, the family will always pick the oldest child to inherit the family lineage. */
    UPDATE person SET inheritor = 1
     WHERE inheritor = 0 
           and year_of_birth <= @current_year - @min_age_of_marriage
           and not exists (SELECT 1 
                             FROM person p2 
                            WHERE p2.person_id < person.person_id
                                  and p2.lineage_id = person.lineage_id
                                  and p2.line_gen >= person.line_gen);

   /* MATCHMAKING POOL #1:
    The chosen inheritors, if they are male, find matches from the pool of non-inheriting women.
   */    

   /* the inheritors are put into the staging_groom table */
   INSERT INTO staging_groom (person_id, ranked, lineage_id, line_gen)
   SELECT person_id,
          row_number() OVER (ORDER BY lineage_id) AS ranked,
          p.lineage_id,
          p.line_gen
     FROM person p
    WHERE sex = 1
          and p.inheritor = 1
          and p.person_id not in (SELECT male_id 
                                    FROM married_pair);

   /* the inheritors find a bride.
      if @sexualSelection is enabled, the inheritors will preferentially pick more attractive matches.
      if not, they are matched at random. */

   WITH
   brides (person_id, attr_pen, mutation_count, surv_pen, year_of_birth)
   AS
   (
   SELECT p.person_id,
          p.attr_pen,
          p.mutation_count,
          p.surv_pen,
          p.year_of_birth
     FROM person p
    WHERE sex = 0
          and p.person_id not in (SELECT female_id 
                                    FROM married_pair)
          and p.year_of_birth <= @current_year - @min_age_of_marriage
          and (p.inheritor = 0 
               or p.inheritor is null)
   )
   INSERT INTO staging_bride (person_id, ranked)
   SELECT person_id, 
          row_number() OVER (ORDER BY CASE WHEN year_of_birth < @current_year - @too_old_for_marriage_age 
                                      THEN 0 
                                      ELSE 1 END ASC, 
                                      CASE WHEN @genetic_Testing_Breeding = 1 
                                      THEN mutation_count 
                                      WHEN @sexual_Selection = 1 THEN attr_pen
                                      ELSE rand(checksum(newid())) END, 
                                      surv_pen ASC) AS ranked 
     FROM brides

   /*Check if family lines need to be merged together.
     should happen in two possible situations: (a) if there are more inheritors than possible matches (I.E. the population is crashing)
     (b) if the total number of family lineages is over the legal limit. (Can only happen if a legal limit was recently added) */

   /* check if there are more inheritors than matches, at least for the grooms. */
   SET @groom_Overflow = (SELECT count(*) 
                            FROM staging_groom) - 
                         (SELECT count(*) 
                            FROM staging_bride)

   /* if @legal_family_count is 0, then that means there is no law limiting the number of families */
   IF @legal_Family_Count <> 0 or @groom_Overflow > 0
   BEGIN

    IF @legal_Family_Count <> 0
    /*get a count of living family lines over the limit */
    SET @family_count = (SELECT count(DISTINCT lineage_id)
                           FROM person
                          WHERE inheritor is not null) - 
                        @legal_Family_Count


    /* @family_count, calculated above, will be negative if the number of families is over the legal limit */
    IF @family_Count > 0 or @groom_Overflow > 0
    BEGIN
    
     /* MATCHMAKING POOL #2:
      if there are too many families, inheritors of each family should marry each other, merging families together. 
      These go into a special matchmaking pool.
      This code is only rarely hit. 
     */

     /*clear out the staging tables to repopulate them with the correct data. 
       since it turns out there were too many families, or at least too many grooms, this data isn't needed. */
     TRUNCATE TABLE staging_bride
     TRUNCATE TABLE staging_groom

     /* the inheritors of the newer families are put into this table if they are men, up to the number of families who need to be merged. */
     INSERT INTO staging_groom (person_id, ranked, lineage_id, line_gen)
     SELECT TOP (CASE WHEN @family_Count > @groom_Overflow 
                      THEN @family_Count 
                      ELSE @groom_Overflow END)
            person_id,
            row_number() OVER (ORDER BY lineage_id) AS ranked,
            p.lineage_id,
            p.line_gen
       FROM person p
      WHERE sex = 1
            and p.person_id not in (SELECT male_id 
                                      FROM married_pair)
            and p.inheritor = 1
      ORDER BY lineage_id DESC

     /* next, the inheritors of the newer families are put into this table if they are women, up to the number of families who need to be merged. */
     INSERT INTO staging_bride (person_id, ranked, lineage_id)
     SELECT TOP (CASE WHEN @family_Count > @groom_Overflow 
                      THEN @family_Count 
                      ELSE @groom_Overflow END)
            person_id,
            row_number() OVER (ORDER BY lineage_id) AS ranked,
            p.lineage_id
       FROM person p
      WHERE sex = 0
            and p.person_id not in (SELECT female_id 
                                      FROM married_pair)
            and p.inheritor = 1
      ORDER BY lineage_id DESC

     /* the match is made provisionally. 
        matchmaking pool #2 is finished */
     INSERT INTO staging_married_pair (female_id,   male_id,    lineage_id,    track_lineages, line_gen)
     SELECT                            f.person_id, m.person_id, m.lineage_id, 1,              m.line_gen
       FROM staging_bride f
       JOIN staging_groom m 
         ON m.ranked = f.ranked

     /* people from the defunct lineage_id are no longer eligible to inherit. 
        for simplicity's sake, it is always the bride's lineage_id that disappears--not that it makes much difference. */
     IF (@family_Count > 0)
     BEGIN
      UPDATE person 
         SET inheritor = null 
       WHERE lineage_id in (SELECT lineage_id 
                              FROM staging_bride)

      UPDATE married_pair 
         SET track_lineages = 0 
       WHERE lineage_id in (SELECT lineage_id 
                              FROM staging_bride)
     end 

     /* the marriage tables are cleared to make room for new matches */
     TRUNCATE TABLE staging_groom;
     TRUNCATE TABLE staging_bride;

     /*the remaining male inheritors are put into this table. 
       Matchmaking pool #1 is recreated. */
     INSERT INTO staging_groom (person_id, ranked, lineage_id, line_gen)
          SELECT person_id,
                 row_number() OVER (ORDER BY lineage_id) AS ranked,
                 p.lineage_id,
                 p.line_gen
            FROM person p
           WHERE sex = 1
                 and p.person_id not in (SELECT male_id 
                                           FROM married_pair 
                                          UNION 
                                         SELECT male_id 
                                           FROM staging_married_pair)
                 and p.inheritor = 1;

     /*the inheritors find a bride from the pool of non-inheriting women.
       Matchmaking pool #1 is recreated. */
     WITH
     brides (person_id, attr_pen, mutation_count, surv_pen, year_of_birth)
     AS
     (
     SELECT p.person_id,
            p.attr_pen,
            p.mutation_count,
            p.surv_pen,
            p.year_of_birth
       FROM person p
      WHERE sex = 0
            and p.person_id not in (SELECT female_id 
                                      FROM married_pair
                                     UNION
                                    SELECT female_id
                                      FROM staging_married_pair)
            and p.year_of_birth <= @current_year - @min_age_of_marriage
            and (p.inheritor = 0 
                 or p.inheritor is null)
     )
     INSERT INTO staging_bride (person_id, ranked)
     SELECT person_id, 
            row_number() OVER (ORDER BY CASE WHEN year_of_birth < @current_year - @too_old_for_marriage_age 
                                             THEN 0 
                                             ELSE 1 END ASC, 
                                        CASE WHEN @genetic_Testing_Breeding = 1 
                                             THEN mutation_count 
                                             WHEN @sexual_Selection = 1 
                                             THEN attr_pen
                                             ELSE rand(checksum(newid())) END, 
                                        surv_pen ASC) AS ranked 
       FROM brides

    END
    
   END
      
   /* the match is made provisionally.
      matchmaking pool #1 is completed */
   INSERT INTO staging_married_pair (female_id,   male_id,     lineage_id,   track_lineages, line_gen)
   SELECT                            f.person_id, m.person_id, m.lineage_id, 1,              m.line_gen
     FROM staging_bride f
     JOIN staging_groom m 
       ON m.ranked = f.ranked
   
   /* marriage tables are cleared to make room for new matches */
   TRUNCATE TABLE staging_groom
   TRUNCATE TABLE staging_bride
      
   /* Matchmaking pool #3:
      the inheritors of each family are put into this table if they are female. */
   INSERT INTO staging_bride (person_id, ranked, lineage_id, line_gen)
   SELECT person_id,
          row_number() OVER (ORDER BY lineage_id) AS ranked,
          p.lineage_id,
          p.line_gen
     FROM person p
    WHERE sex = 0
          and p.person_id not in (SELECT female_id 
                                    FROM married_pair 
                                   UNION 
                                  SELECT female_id 
                                    FROM staging_married_pair)
          and p.inheritor = 1;

   /* the inheritors find a groom from the pool of non-inheriting men. */ 
   WITH
   grooms (person_id, parents_id, attr_pen, mutation_count, surv_pen, year_of_birth)
   AS
   (
   SELECT p.person_id,
          p.parents_id,
          p.attr_pen,
          p.mutation_count,
          p.surv_pen,
          p.year_of_birth
     FROM person p
    WHERE p.sex = 1
          and p.person_id not in (SELECT male_id 
                                    FROM married_pair 
                                   UNION 
                                  SELECT male_id 
                                    FROM staging_married_pair)   
          and p.year_of_birth <= @current_year - @min_age_of_marriage
          and (p.inheritor = 0 or p.inheritor is null)
   )   
   INSERT INTO staging_groom (person_id, ranked)
   SELECT person_id, 
          row_number() OVER (ORDER BY CASE WHEN year_of_birth < @current_year - @too_old_for_marriage_age 
                                           THEN 0 
                                           ELSE 1 END ASC, 
                                      CASE WHEN @genetic_Testing_Breeding = 1 
                                           THEN mutation_count 
                                           WHEN @sexual_Selection = 1 
                                           THEN attr_pen
                                           ELSE rand(checksum(newid())) END ASC, 
                                       surv_pen ASC) AS ranked
     FROM grooms;

   /* the matches are made provisionally
      Matchmaking pool #3 is completed */
   INSERT INTO staging_married_pair (female_id,   male_id,     lineage_id,   track_lineages, inheritor_sex, line_gen)
   SELECT                            f.person_id, m.person_id, f.lineage_id, 1,              0,             f.line_gen
     FROM staging_bride f
     JOIN staging_groom m 
       ON m.ranked = f.ranked
   
   /* Tables are cleared out to make room for new matches. */
   TRUNCATE TABLE staging_bride
   TRUNCATE TABLE staging_groom

  END                                
  

  /* MATCHMAKING POOL #4:
   For extra couples who don't inherit and don't marry an inheritor. 
   If family lineages is off, this is the only matchmaking pool.
   If family lineages is on, this group marries slightly later.
  */

  /* Skip this matchmaking pool if there were inheritors who couldn't find normal matches.
     also skip this matchmaking pool if family count is already at the cap and unlicensed families aren't allowed. */
  IF @groom_Overflow <= 0 and (@family_Count < 0 or @unlicensed_Families = 1 or @unlicensed_Families is null)
  BEGIN

   /* The staging table is populated with eligible women */
   WITH
   brides (person_id, attr_pen, mutation_count, surv_pen, year_of_birth)
   AS
   (
   SELECT p.person_id as person_id,
          p.attr_pen,
          p.mutation_count,
          p.surv_pen,
          p.year_of_birth
     FROM person p
    WHERE sex = 0
          and p.person_id not in (SELECT female_id 
                                    FROM married_pair 
                                   UNION 
                                  SELECT female_id 
                                    FROM staging_married_pair)  
          and p.year_of_birth <= @current_year - CASE WHEN @track_family_lineages = 0 
                                                      THEN @min_age_of_marriage 
                                                      ELSE @free_marriage_age END
          and (p.inheritor = 0 
               or p.inheritor is null)
   )
   INSERT INTO staging_bride (person_id, ranked)
   SELECT person_id, row_number() OVER (ORDER BY CASE WHEN year_of_birth < @current_year - @too_old_for_marriage_age 
                                                      THEN 0 
                                                      ELSE 1 END ASC, 
                                                 CASE WHEN @genetic_Testing_Breeding = 1 
                                                      THEN mutation_count 
                                                      WHEN @sexual_Selection = 1 THEN attr_pen
                                                      ELSE rand(checksum(newid())) END ASC, 
                                                 surv_pen ASC) AS ranked 
   FROM brides;

   /* The staging table is populated with eligible men */
   WITH
   grooms (person_id, attr_pen, mutation_count, surv_pen, year_of_birth)
   AS
   (
   SELECT p.person_id,
          p.attr_pen,
          p.mutation_count,
          p.surv_pen,
          p.year_of_birth
     FROM person p
    WHERE sex = 1
          and p.person_id not in (SELECT male_id 
                                    FROM married_pair 
                                   UNION
                                  SELECT male_id 
                                    FROM staging_married_pair)
          and p.year_of_birth <= @current_year - CASE WHEN @inheritance_selection = 0 
                                                      THEN @min_age_of_marriage 
                                                      ELSE @free_marriage_age END
          and (p.inheritor = 0 
               or p.inheritor is null)
   )
   INSERT INTO staging_groom (person_id, ranked)
   SELECT person_id, row_number() OVER (ORDER BY CASE WHEN year_of_birth < @current_year - @too_old_for_marriage_age 
                                                      THEN 0 
                                                      ELSE 1 END ASC, 
                                                 CASE WHEN @genetic_Testing_Breeding = 1 
                                                      THEN mutation_count 
                                                      WHEN @sexual_Selection = 1 
                                                      THEN attr_pen
                                                      ELSE rand(checksum(newid())) END ASC, 
                                                 surv_pen ASC) AS ranked 
     FROM grooms
   
   /* the total number of new matches is limited by the @percentage_matching setting, 
      or by the number of new families that can be legally created if a law limiting families exists.*/
   IF @unlicensed_Families = 0
    SET @new_Matches = -@family_Count
   ELSE
    SET @new_Matches = 1+(@percentage_Matching*(SELECT count(*) 
                                                  FROM staging_bride))/100


   /*the matches are made provisionally 
     Matchmaking pool #4 is completed.*/
   INSERT INTO staging_married_pair (female_id, male_id, lineage_id, track_lineages)
   SELECT TOP (@new_Matches) 
          f.person_id as female_id, 
          m.person_id as male_id,
          0 as lineage_id,
          0 as track_lineages
   FROM staging_bride f
   JOIN staging_groom m on m.ranked = f.ranked

  END
    
  /*remove incestuous marriages, except when population is extremely low. */
  IF (SELECT Count(*) 
        FROM married_pair) > 100
   DELETE FROM staging_married_pair 
    WHERE pair_id in (SELECT pair_id 
                        FROM staging_married_pair smp
                        JOIN person m 
                          ON m.person_id = smp.male_id
                        JOIN person f 
                          ON f.person_id = smp.female_id
                       WHERE m.lineage_id = f.lineage_id)
  
  /*if there are not enough licensed families, new family licenses are issued to new non-inheriting couples. */
  IF @family_Count < 0 and @track_Family_Lineages = 1
   UPDATE staging_married_pair
      SET lineage_id = pair_id,
          track_lineages = 1,
          line_gen = 1
    WHERE pair_id in (SELECT TOP (-@family_count) 
                             pair_id 
                        FROM staging_married_pair
                       WHERE lineage_id = 0 
                       ORDER BY pair_id ASC)
  
  /* the marriages are finalized */
  INSERT INTO married_pair (pair_id, female_id, male_id, lineage_id, line_gen, desired_children, track_lineages)
  SELECT pair_id, 
         female_id, 
         male_id, 

         CASE smp.track_lineages
         WHEN 0 THEN 0
         ELSE smp.lineage_id END AS lineage_id,

         CASE smp.track_lineages 
         WHEN 0 THEN 0
         ELSE smp.line_gen
          END AS line_gen, 
                 
         CASE WHEN smp.track_lineages = 1 or @legal_family_count is NULL
              THEN floor(rand(checksum(newid()))*(@max_cultural_child_count+1-@min_cultural_child_count))+@min_cultural_child_count 
              ELSE floor(rand(checksum(newid()))*(@max_unlicensed_child_count+1-@min_unlicensed_child_count))+@min_unlicensed_child_count 
               END AS desired_children,

         CASE smp.lineage_id 
         WHEN 0 THEN 0 
         ELSE 1 END AS track_lineages

    FROM staging_married_pair smp;
      
  /* for performance reasons, the genetic data of the pair is combined into the married_pair_mutations table to limit calls to the very large mutation table. */
  /* First, the mother's mutations are retrieved from the mutation table */      
  WITH
  mothers_mutations (pair_id, mutation_id, mothers_variant, fathers_variant)
  AS
  (
  SELECT pair_id, 
         mutation_id, 
         mothers_variant, fathers_variant
    FROM staging_married_pair mp
    JOIN mutation m 
      ON mp.female_id = m.person_id
  ),
  /* the father's mutations are retrieved */
  fathers_mutations (pair_id, mutation_id, mothers_variant, fathers_variant)
  AS
  (
  SELECT pair_id, 
         mutation_id, 
         mothers_variant, 
         fathers_variant
    FROM staging_married_pair mp
    JOIN mutation m 
      ON mp.male_id = m.person_id
  )
  /* the mutation data is combined together and inserted into the married_pair_mutation table.
     This way, the mutation table only needs to be read once per marriage rather than once per child, saving CPU cycles. */
  INSERT INTO married_pair_mutation (pair_id, mutation_id,females_mothers_variant, females_fathers_variant, males_mothers_variant, males_fathers_variant)
  SELECT IsNull(f.pair_id,m.pair_id),
         IsNull(f.mutation_id, m.mutation_id),
         IsNull(f.mothers_variant,0),
         IsNull(f.fathers_variant,0),
         IsNull(m.mothers_variant,0),
         IsNull(m.fathers_variant,0)
    FROM mothers_mutations f
    FULL OUTER JOIN fathers_mutations m
      ON m.pair_id = f.pair_id 
         and m.mutation_id = f.mutation_id;
  
  /* the marriage staging tables are again cleared */
  truncate table staging_bride
  truncate table staging_groom


  /* Remembers the next identity seed from staging_married_pair before truncating the table. */
  SET @save_identity = ident_current('staging_married_pair')+1;

  /* If the value is close to causing an integer overflow error, reset it to 1. */
  IF @save_identity > 2000000000
     SET @save_identity = 1;

  /* Clears the staging_married_pair table of data in preparation for the next run */
  TRUNCATE TABLE staging_married_pair;

  /* Resets the identity seed of staging_person to the proper value. */
  DBCC checkident('staging_married_pair', RESEED, @save_identity);
  
  /* The loop is finished. increment the year to prepare for the next loop */
  SET @current_year = @current_year + 2

  /* report data on population, mutations per person, average survpen, date, and child mortality statistics every @report_frequency number of years. */
  IF @current_year%@report_Frequency = 0
  BEGIN
   
   SELECT  /* reports the current year */
           @current_year as [date],

           /* the average of the total population since the last report */
           Avg(current_pop + births) AS total_population, 

           /* the average number of mutations per person.
              TODO: add this value to the statistics table and average it like the others. */
           (SELECT Avg(mutation_count) 
              FROM person) AS deleterious_mutations_per_person, 
           
           /* the average adult fitness detriment.
              TODO: Figure out whether this is even a good metric. 
              If it is, add it to the statistics table and average it like the others. */
           100 *
           cast((SELECT Avg(success_pen) 
                   FROM person 
                  WHERE year_of_birth < @current_year-18) AS MONEY) /
           @surv_comparison_small AS avg_adult_fitness_detriment,

           /* the average variance in fertility... how much each person's fertility differs from the average on average. */
           Avg(fertility_variance) as fertility_variance,
           
           /* the average child mortality over the period */
           Avg(child_mortality) as child_mortality
      
      FROM statistics_table
     WHERE current_year between @current_year-20 and @current_year-2

  END 
    
 END